﻿/*
VERSION:		2.2

2.2		added getBitmap() to the output object  +  Returned object emits real events
2.1		added onAnimDone() callback function into returned loader object
2.0		detects successful linkage even if movieClip is empty
1.9		waitForLoad() & waitForAnim()	chain-able monads.  Reliable & instant movieClip replacement with onUnload() call.
1.8		used "function" prefix to allow local scoping
1.7		output "loader" includes a reference to the new movieClip as a variable named "content"
1.6		onLoadInit() passes a reference to the loaded movieClip
1.5		added "isVisible" flag to set the default visibility of the loaded image  (doesn't apply when result is BitmapData)


EVENTS		(emitted by return_obj)
	onLoadInit				runs when the file is successfully loaded
	complete					exactly the same as onLoadInit
	onLoadError				runs if the file cannot be loaded
	onAnimDone				runs the first time the loaded file reaches its last animation frame
	
	
RETURN VALUES:
	{
		onLoadInit()		Externally-defined callback function, which runs when the file is successfully loaded
		onLoadError()		Externally-defined callback function, which runs if the file cannot be loaded
		onAnimDone()		Externally-defined callback function, which runs the first time the loaded file reaches its last animation frame
		content					Reference to the internally created/loaded movieClip, which always contains an animation or a displayed bitmap
		waitForLoad()		Adds functions to a queue which fires when the file has been loaded, or instantly if the file has already loaded
		waitForAnim()		Adds functions to a queue which fires the first time the loaded file reaches its last animation frame
	}
	
	
DESCRIPTION:
	Creates a movieClip as specified & loads an image, SWF, or bitmapData into it.
	The returned "loader" has chain-able functions:  waitForLoad  &  waitForAnim  (they're not promises, but they behave sort of like that)
	
	
DEFAULT RESULT:
	this.image_0
	
	
USAGE:
	#include "loadImage.as"
	loader = loadImage("test.jpg");
	
	loader
	.waitForLoad( success )
	.waitForAnim( onAnimDone )
	.waitForLoad( success2 );
	
	function success( mc, getBitmap ){
		trace("success");
	}
	loader.onLoadInit = success;
	loader.onLoadError = function(){
		trace("fail");
	}
	
	OR
	
	react.to("complete").from(loader).then = success;
	
	function onAnimDone( mc ){}
*/
function loadImage(file, target, newName, newDepth, isVisible)
{
	///////////////////////////////////////////////////////////////////////////////////////
	// dependencies
	#include "nextDepth.as"
	
	
	
	// END: dependencies
	///////////////////////////////////////////////////////////////////////////////////////
	// shared variables
	//if( isLoaded )					waitForLoad_trigger();
	//if( isDoneAnimating )		waitForAnim_trigger();
	var return_obj;
	var isLoaded = false;
	var isDoneAnimating = false;
	var waitForLoad_queue = [];
	var waitForAnim_queue = [];
	var loadedBitmap = null;		// direct bitmap / cached bitmap snapshot of loaded content
	var detectTargetUnload = {};
	var detectLastFrame_interval = null
	var cleanupInterval = null;
	
	
	// END: shared variables
	///////////////////////////////////////////////////////////////////////////////////////
	// helper functions
	function once( func ){
		var done = false;
		return function () {
			return done ? void 0 : ((done = true), func.apply(this, arguments));
		}
	}// once()
	
	
	function unload(){
		if(cleanupInterval !== null){
			clearInterval( cleanupInterval );
			cleanupInterval = null;
		}
		// announce that unload has occurred
		return_obj["broadcastMessage"]("unload");
		//
		removeEvents();
	}// unload()
	var unload = once( unload );
	
	
	function makeOnUnload( placeIntoThis ){
		if( placeIntoThis.onUnload )		return;		// do not replace an existing onUnload() function
		placeIntoThis.onUnload = unload;
	}// makeOnUnload()
	
	
	// clean-up events when target unloads  (target emits an "unload" event)
	function removeEvents(){
		// disable unload event
		detectTargetUnload.unload = undefined;		delete detectTargetUnload.unload;
		target.removeEventListener( "unload", detectTargetUnload );
		target.removeListener( detectTargetUnload );
		// remove ALL listeners from return_obj
		for(var L in return_obj["_listeners"]){
			var thisListener = return_obj["_listeners"][L];
			return_obj["removeListener"]( thisListener );
		}// for:  each listener
		// de-initialize AsBroadcaster
		return_obj["_listeners"] = undefined;		delete return_obj["_listeners"];
		return_obj["addListener"] = undefined;		delete return_obj["addListener"];
		return_obj["removeListener"] = undefined;		delete return_obj["removeListener"];
		return_obj["broadcastMessage"] = undefined;		delete return_obj["broadcastMessage"];
		target.removeListener( return_obj );
	}// removeEvents()
	
	
	function getBitmap(){
		if( !return_obj.content )		return;		// if there is no content => undefined
		if(!loadedBitmap){
			// take a snapshot of the loaded content
			var wid = return_obj.content._width;
			var hei = return_obj.content._height;
			// movieClips can contain nothing, so avoid outputting zero-size bitmaps
			if( !wid )		wid = 1;
			if( !hei )		hei = 1;
			loadedBitmap = new flash.display.BitmapData( wid,hei ,true,0);
			loadedBitmap.draw( return_obj.content );
		}// if:  no loadedBitmap exists for this content yet
		return loadedBitmap;
	} // getBitmap()
	
	
	function detectLastFrame( anim_mc ){
		// detect last animation frame
		var detectLastFrame = {
			loop:function(){
				//var anim_mc = target[newName];
				if( anim_mc._currentframe==0  ||  anim_mc._currentframe==undefined )
				{// if:  this clip was deleted
					// stop checking
					clearInterval( detectLastFrame_interval );
					// clear the queue
					waitForAnim_queue.splice(0, waitForAnim_queue.length);
				}// if:  this clip was deleted
				else if(anim_mc._currentframe == anim_mc._totalframes)
				{// if:  this clip is on its last animation frame
					// stop checking
					clearInterval( detectLastFrame_interval );
					// trigger the queue
					isDoneAnimating = true;
					waitForAnim_trigger();
					return_obj.onAnimDone( return_obj.content, return_obj.getBitmap );
					return_obj.broadcastMessage("onAnimDone", return_obj.content, return_obj.getBitmap );
					delete return_obj.onAnimDone;
					// clear the queue
					waitForAnim_queue.splice(0, waitForAnim_queue.length);
				}// if:  this clip is on its last animation frame
			}// loop()
		}// detectLastFrame obj
		detectLastFrame_interval = setInterval( detectLastFrame.loop, 33 );
		detectLastFrame.loop();
	}// detectLastFrame()
	detectLastFrame = once( detectLastFrame );
	
	
	function contentExists(){
		return (return_obj.content._currentframe !== undefined);		// exist => true,  not-exist => false
	}// contentExists()
	
	
	// END: helper functions
	///////////////////////////////////////////////////////////////////////////////////////
	// create + populate the return object
	// ... including its chain-able monad callbacks
	function waitForLoad_trigger(){
		for(var i=0; i<waitForLoad_queue.length; i++){
			//waitForLoad_queue[i]( return_obj.content );
			waitForLoad_queue[i].apply( return_obj.content, [return_obj.content, return_obj.getBitmap] );
		}// for:  each callback in thie queue
		// empty the queue
		waitForLoad_queue.splice(0, waitForLoad_queue.length);
	}// waitForLoad_trigger()
	function waitForAnim_trigger(){
		for(var i=0; i<waitForAnim_queue.length; i++){
			//waitForAnim_queue[i]( return_obj.content );
			waitForAnim_queue[i].apply( return_obj.content, [return_obj.content, return_obj.getBitmap] );
		}// for:  each callback in thie queue
		// empty the queue
		waitForAnim_queue.splice(0, waitForAnim_queue.length);
	}// waitForAnim_trigger()
	
	
	
	// monad interface
	return_obj = {
		onLoadInit:function(){},
		onLoadError:function(){},
		onAnimDone:function(){},
		content: undefined,
		waitForLoad: function( newCallback ){
			if(isLoaded){
				newCallback.apply( return_obj.content, [return_obj.content, getBitmap] );
			}else{
				waitForLoad_queue.push( newCallback );
			}
			return return_obj;		// return parent object, making this a chain-able monad
		}, // waitForLoad()
		waitForAnim: function( newCallback ){
			if( isDoneAnimating ){
				newCallback.apply( return_obj.content, [return_obj.content, getBitmap] );
			}else{
				waitForAnim_queue.push( newCallback );
			}
			return return_obj;		// return parent object, making this a chain-able monad
		}, // waitForAnim()
		getBitmap: getBitmap,
		unload: unload,
		removeEvents: removeEvents
	}// return_obj
	
	// Allow return_obj to emit events
	AsBroadcaster.initialize( return_obj );
	
	
	
	// END: create + populate the return object
	///////////////////////////////////////////////////////////////////////////////////////
	// detect unload => clean-up  (A)
	// add the ability to react to "unload" events in general
	detectTargetUnload.unload = unload;
	
	// react to "unload" from the new image's parent
	if(target.addEventListener){
		target.addEventListener( "unload", detectTargetUnload );
	}else if(target.addListener){
		target.addListener( detectTargetUnload );
	}
	
	// periodically check if the target goes missing  (last-resort fail-safe in case onUnload() fails to fire)
	var randStartDelay = Math.floor( Math.random() * 300 );		// this prevents lots of loadImage's from all checking at the same time
	setTimeout(function(){
		cleanupInterval = setInterval(function(){
			var exists = contentExists();
			if(exists)		return;		// content exists => continue checking
			
			// exists no longer exists
			unload();
		}, 3000);
	}, randStartDelay);
	
	
	
	
	// END: detect unload => clean-up
	///////////////////////////////////////////////////////////////////////////////////////
	// definitions are done,
	// Begin actual program
	if(isVisible === undefined)		isVisible = true;
	// resolve optional parameters
	var target = target || this;
	var newDepth = (newDepth!=undefined) ? newDepth : nextDepth(target);
	var newName = newName || "image_"+newDepth;
	
	
	// if:  replacing an existing movieClip
	if(newName  &&  target[newName]){
		target[newName].onUnload();
		delete target[newName].onUnload;		// prevent unload delay  &  double-calls
		var tempDepth = nextDepth( target );
		if( target[newName].getDepth() < 0 )		target[newName].swapDepths( tempDepth );		// guarantee that removeMovieClip() will work
		target[newName].unloadMovie();		target[newName].removeMovieClip();
	}// if:  replacing an existing movieClip
	
	
	
	// try movieClip linkage  &  create container
	var newImage = target.attachMovie( file, newName, newDepth, {_visible:isVisible} );
	var newImage = target[newName];
	//if( newImage._width > 0  &&  newImage != target)
	if( target.getInstanceAtDepth(newDepth) !== undefined  &&  newImage != target )
	{// if:  movieClip loaded
		//trace("* MovieClip linkage");
		loadedBitmap = null;
		return_obj.content = newImage;
		isLoaded = true;
		// detect unload => clean-up  (B)
		makeOnUnload( return_obj.content );
		waitForLoad_trigger();
		detectLastFrame( target[newName] );
		setTimeout( function(){
			return_obj["onLoadInit"]( return_obj.content, return_obj.getBitmap );
			return_obj.broadcastMessage("onLoadInit", return_obj.content, return_obj.getBitmap );
			return_obj.broadcastMessage("complete", return_obj.content, return_obj.getBitmap );
		}, 0 );
	}// if:  movieClip loaded
	
	
	
	// try bitmapData linkage or reference
	else
	//if(	newImage == target  ||
	//	 	newImage._width == undefined  ||
	//		newImage._width == 0)
	{// if:  no movieClip loaded
		// create container
		var newImage = target.createEmptyMovieClip(newName, newDepth);
		newImage._visible = isVisible;
		// // direct bitmapData
		if(file.generateFilterRect != undefined)
			var image_pic = file;
		// bitmap linkage
		if(file.generateFilterRect == undefined)
			var image_pic = flash.display.BitmapData.loadBitmap( file );
		
		newImage.attachBitmap(image_pic, 0, true, true);
		//if( newImage._width > 0 )
		if( newImage.getInstanceAtDepth(0) !== undefined )
		{// if:  bitmap loaded
			//trace("* Bitmap linkage");
			loadedBitmap = image_pic;
			return_obj.content = newImage;
			isLoaded = true;
			// detect unload => clean-up  (B)
			makeOnUnload( return_obj.content );
			waitForLoad_trigger();
			// bitmaps don't animate, so the animation is now done  (last frame is already displayed)
			isDoneAnimating = true;
			waitForAnim_trigger();
			return_obj.onAnimDone( return_obj.content, return_obj.getBitmap );
			return_obj.broadcastMessage("onAnimDone", return_obj.content, return_obj.getBitmap );
			delete return_obj.onAnimDone;
			setTimeout( function(){
				return_obj["onLoadInit"]( return_obj.content, return_obj.getBitmap );
				return_obj.broadcastMessage("onLoadInit", return_obj.content, return_obj.getBitmap );
				return_obj.broadcastMessage("complete", return_obj.content, return_obj.getBitmap );
			}, 0 );
		}// if:  bitmap loaded
		
		
		
		// try loading external file
		else
		//if(	newImage._width == undefined  ||
		//newImage._width == 0)
		{// if:  no bitmap loaded
			// external file
			var loader = new MovieClipLoader();
			loadedBitmap = null;
			return_obj.content = newImage;
			
			loader.onLoadInit = function( loadedClip ){
				//trace("* File loaded");
				newImage._visible = isVisible;
				loadedBitmap = null;
				return_obj.content = loadedClip;
				isLoaded = true;
				// detect unload => clean-up  (B)
				makeOnUnload( return_obj.content );
				waitForLoad_trigger();
				detectLastFrame( loadedClip );
				setTimeout( function(){
					return_obj["onLoadInit"]( return_obj.content, return_obj.getBitmap );
					return_obj.broadcastMessage("onLoadInit", return_obj.content, return_obj.getBitmap );
					return_obj.broadcastMessage("complete", return_obj.content, return_obj.getBitmap );
				}, 0 );
			}// onLoadInit()
			loader.onLoadInit = once( loader.onLoadInit );
			loader.onLoadError = function(){
				// clear the current queue because it won't be needed
				waitForLoad_queue.splice(0, waitForLoad_queue.length);
				waitForAnim_queue.splice(0, waitForAnim_queue.length);
				return_obj["onLoadError"]();
				return_obj.broadcastMessage("onLoadError" );
			}// onLoadError()
			loader.onLoadError = once( loader.onLoadError );
			loader.loadClip( file, newImage );
			
			if(isLoaded)
			{// if:  instant load
				// pretend that it's not instant
				setTimeout( function(){
					loader.onLoadInit( target[newName] );
				}, 0);
			}// if:  instant load			
			//return return_obj;
			//return loader;
		}// if:  no bitmap loaded
	}// if:  no movieClip loaded
	
	
	
	
	return return_obj;
}// loadImage()